<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Models\CommitteeMember;
use App\Models\Slide;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Inertia\Inertia;
use Inertia\Response;

class PublicController extends Controller
{
    // --- Helper for Prayer Times ---
    private function getPrayerTimes()
    {
        // Try to get from API or Database, fallback to static
        try {
            // Get location from settings
            $latitude = setting('location_latitude', '-6.200000');
            $longitude = setting('location_longitude', '106.816666');
            
            $date = now()->format('d-m-Y');
            $response = Http::timeout(3)->get("http://api.aladhan.com/v1/timings/$date", [
                'latitude' => $latitude,
                'longitude' => $longitude,
                'method' => 11, // Majlis Ugama Islam Singapura, close to Indonesia standard
            ]);

            if ($response->successful()) {
                $timings = $response->json('data.timings');
                return [
                    'Subuh' => $timings['Fajr'],
                    'Dzuhur' => $timings['Dhuhr'],
                    'Ashar' => $timings['Asr'],
                    'Maghrib' => $timings['Maghrib'],
                    'Isya' => $timings['Isha'],
                    'date' => now()->translatedFormat('l, d F Y')
                ];
            }
        } catch (\Exception $e) {
            // Fallback logic below
        }

        // STATIC FALLBACK (If API fails) to prevent "Calculating..." loop
        return [
            'Subuh' => '04:30',
            'Dzuhur' => '12:05',
            'Ashar' => '15:20',
            'Maghrib' => '18:10',
            'Isya' => '19:25',
            'date' => now()->translatedFormat('l, d F Y')
        ];
    }

    public function structure(): Response
    {
        $committee = CommitteeMember::where('is_active', true)
            ->orderBy('division')
            ->orderBy('order')
            ->get()
            ->groupBy('division');
        
        return Inertia::render('Public/Struktur', [
            'committee' => $committee,
        ]);
    }
    
    public function keuangan(): Response
    {
        // Get latest 50 approved transactions
        $transactions = Transaction::with('verifiedBy', 'approvedBy')
            ->approved()
            ->latest('date')
            ->take(50)
            ->get()
            ->map(fn($t) => [
                'id' => $t->id,
                'uuid' => $t->uuid,
                'type' => $t->type,
                'category' => $t->category,
                'amount' => $t->amount,
                'formatted_amount' => $t->formatted_amount,
                'description' => $t->description,
                'date' => $t->date->format('d M Y'),
                'proof_url' => $t->proof_url,
                'verified_by_name' => $t->verifiedBy?->name,
            ]);
        
        // Financial summary
        $totalIncome = Transaction::income()->approved()->sum('amount');
        $totalExpense = Transaction::expense()->approved()->sum('amount');
        $balance = $totalIncome - $totalExpense;
        
        return Inertia::render('Public/Keuangan', [
            'transactions' => $transactions,
            'summary' => [
                'income' => $totalIncome,
                'expense' => $totalExpense,
                'balance' => $balance,
                'formatted_income' => 'Rp ' . number_format($totalIncome, 0, ',', '.'),
                'formatted_expense' => 'Rp ' . number_format($totalExpense, 0, ',', '.'),
                'formatted_balance' => 'Rp ' . number_format($balance, 0, ',', '.'),
            ],
        ]);
    }
    
    public function aset(): Response
    {
        $assets = Asset::latest()->get();
        
        return Inertia::render('Public/Aset', [
            'assets' => $assets,
        ]);
    }
    
    public function galeri(): Response
    {
        $slides = Slide::where('is_active', true)
            ->orderBy('order')
            ->get();
        
        return Inertia::render('Public/Galeri', [
            'slides' => $slides,
        ]);
    }

    // --- NEW METHODS ---

    public function tentang(): Response
    {
        return Inertia::render('Public/Tentang');
    }

    public function jumat(): Response
    {
        // Mock data for now, can be replaced with DB call later
        $schedule = [
            'date' => $this->getNextFriday(),
            'khatib' => 'Ustadz Ahmad Dahlan, Lc',
            'imam' => 'H. Muhammad Ridwan',
            'muadzin' => 'Sdr. Ali Rahman',
            'bilal' => 'Sdr. Umar'
        ];

        return Inertia::render('Public/Jumat', [
            'schedule' => $schedule,
            'prayerTimes' => $this->getPrayerTimes()
        ]);
    }

    public function jadwal(): Response
    {
        return Inertia::render('Public/Jadwal', [
            'prayerTimes' => $this->getPrayerTimes()
        ]);
    }

    public function agenda(): Response
    {
        // Real active agendas
        $agendas = \App\Models\Agenda::active()
            ->whereDate('date', '>=', now())
            ->orderBy('date', 'asc')
            ->orderBy('time', 'asc')
            ->get()
            ->map(function ($agenda) {
                return [
                    'id' => $agenda->id,
                    'title' => $agenda->title,
                    'date' => \Carbon\Carbon::parse($agenda->date)->translatedFormat('l, d F Y'),
                    'time' => \Carbon\Carbon::parse($agenda->time)->format('H:i') . ' WIB',
                    'location' => $agenda->location,
                    'description' => $agenda->description ?? '-',
                    // 'speaker' is not in DB yet, but could be added later or part of description
                ];
            });

        return Inertia::render('Public/Agenda', [
            'agendas' => $agendas
        ]);
    }

    private function getNextFriday()
    {
        $now = now();
        if ($now->dayOfWeek === Carbon::FRIDAY) {
            return $now->translatedFormat('d F Y');
        }
        return $now->next(Carbon::FRIDAY)->translatedFormat('d F Y');
    }
}
